/*=========================================================
	MODULE zFileImport
	----------------------------------------------------
	Fonctions d'import fichier Excel ou CSV
	
	V1.0 :
=========================================================*/
import fs from 'node:fs';
import XLSX from "node-XLSX";
import * as Date from 'date-and-time';
import * as normFunctions from './normalization.mjs';

// Variables globales du module
let dbPool = null;

export function fileImportuseDbPool(pool) {
	dbPool = pool;
}

// Lecture des structures d'import attendues
import importModels from '../data/fileImport.json' with { type: "json" };

export async function importFile(importData) {
	/* Import d'un fichier vers une table de la base de données
		Contenu de l'objet 'importData'
			importData.filePath	: Chemin d'accès au fichier
			importData.sheetName	: Nom de la feuille contenant les données à importer (par défaut première feuille)
			importData.model		: Modèle d'import demandé
			importData.dbTable	: Table d'import des données collectées
	*/
	return new Promise(async (resolve, reject) => {
		console.log("importData:", importData);
		
		// Vérification de l'existance du fichier
		console.log("> Vérification de la disponibilité du fichier");
		if(!fs.existsSync(importData.filePath)) return reject(`!!! Le fichier '${filePath}' est introuvable`);
		const fileName = importData.filePath.split("\\").pop().split("/").pop();
		//  Recherche du modèle
		console.log("> Recherche du modèle d'import");
		const model = importData.model ? importModels.find(item => item.model==importData.model) : null;
		if(!model) return reject(`!!! Modèle de fichier '${importData.model}' on connu ou inexistant dans les paramètres`);
		// Mode d'import
		const importMode = importData.mode ? importData.mode.toUpperCase() : "OVERWRITE";
		if(!["OVERWRITE", "UPDATE", "IGNORE", "FROM SCRATCH"].includes(importMode))  return reject(`!!! Mode d'import "${importData.mode}" inconnu`);
		// Récupération de la liste des tables de la base de données
		const dbTables = await dbPool.getTables();
		if(!dbTables.includes(importData.dbTable)) return reject(`!!! La table d'import "${importData.dbTable}" n'existe pas dans la base de données "${dbPool.database}"`);

		// lecture du fichier 
		console.log("> Lecture du fichier");
		const fileFullContent = XLSX.parse(fs.readFileSync(importData.filePath));
		const useSheet = (importData.sheetName)? importData.sheetName : null;
		const sheet = (fileFullContent.some(obj => obj.name=useSheet))? fileFullContent.find(sheet => sheet.name==useSheet) : fileFullContent[0];
		const sheetData = sheet.data;
		const sheetHeader = sheetData[0];

		// Identication du schéma
		console.log("> Identification du schéma");
		let schema = null;
		try {
			schema = await getSchema(sheetHeader, model)
			console.log(` ... Schéma identifié [${schema.name}]`);
		} catch(err) {
			return reject(err);
		}

		// Table d'import
		let importTable = null;
		if(model.normalization) {
			// Controle de la présence de la fonction de normalisation
			if(typeof normFunctions[model.normalization.function]!='function') return reject(`!!! La fonction de normalisation des données "${model.normalization.function}" est inconnue`);
			console.log(" - Import via une table intermédiaire");
			// Définition et nettoyage de la table d'import
			importTable = model.normalization.tbImport;
			if(!dbTables.includes(importData.dbTable)) return reject(`!!! La table d'import temporaire"${importTable}" n'existe pas dans la base de données "${dbPool.database}"`);
			console.log(` ... Purge de la table "${importTable}"`);
			await dbPool.executeQuery(`TRUNCATE TABLE ${importTable}`);
		} else {
			importTable = importData.dbTable;			
		}
		// Purge de la table si mode 'FROM SCRATCH'
		if(importMode=="FROM SCRATCH" && !model.normalization) {
			console.log(` ... Purge de la table "${importTable}"`);
			await dbPool.executeQuery(`TRUNCATE TABLE ${importTable}`);
		}

		// Parcours des données pour import
		let index = 1;
		console.log(` ... Import des données dans "${importTable}"`);
		while(index<sheetData.length) {
			// Construction de l'objet 'data' constitué des données de l'entête comme clé et des données de la ligne en cours comme valeurs
			const data = Object.fromEntries(sheetHeader.map((key, i) => [key, sheetData[index][i]]));
			await formatImportData(data, schema, importTable, (model.normalization)? "FROM SCRATCH" : importMode);
			index++;
		}

		if(model.normalization) {
			// Transfert de la table/objet intermédiaire dans la table finale selon la fonction de normalisation
			const normFunction = model.normalization.function;
			normFunctions[normFunction](dbPool, importTable)
			.then(() => { return resolve(`Fichier "${fileName}" a été importé avec succès`) })
			.catch((err) => { return reject(err)});
		}

		resolve(`Fichier "${fileName}" a été importé avec succès`);
	});
}

async function formatImportData(data, schema, dataTable, mode) {
	return new Promise(async (resolve) => {
		var dataSet = [];
		let value;
		
		// Pour chaque élément de la structure d'import
		schema.structure.forEach(element => {
			// Lecture de la donnée à trraiter
			if(Object.hasOwn(element, "label")) {
				value = data[element.label];
			}
			else if(Object.hasOwn(element, "value")) value = element.value;
		
			// Mise en forme de la donnée
			if(element.format) {
				switch(element.format) {
					case "dateEN", "dateISO":
						if(!Date.isValid(value)) value = null;
						break;
					case "dateFR":
						value = Date.transform(value, 'DD/MM/YYYY', "YYYY-MM-DD");
						if(! Date.isValid(value)) value = null;
						break;
					case "datetimeFR":
						value = Date.transform (value, 'DD/MM/YYYY HH:mm:ss', 'YYYY-MM-DD HH:mm:ss');
						if(!Date.isValid(value, 'YYYY-MM-DD HH:mm:ss')) value = null;
						break;
					case "binary":
						value = (value==element.option) ? 1 : 0;
						break;
					case "binary0":
						value = (value==element.option) ? 0 : 1;
						break;
					case "num":
						value = value;
						break;
				}
			}
			// Transformation de la donnée
			if(element.transform) {
				switch(element.transform) {
					case "lowercase":
						value = value.toLowerCase();
						break;
					case "uppercase":
						value = value.tiUpperCase();
						break;
				}
			}
			dataSet[element.field] = value;

		})
		
		var existingDataset = null;
		// Recherche d'un enregistrement correspondant
		if(schema.index && mode!="FROM SCRATCH") {
			var critera = [];
			schema.index.forEach((index) => {
				critera.push(`${index}='${dataSet[index]}'`);
			});
			const querySearch = "SELECT "+Object.keys(dataSet).join(", ")+" FROM "+dataTable+ " WHERE "+critera.join(" AND ");
			//console.log(querySearch);
			existingDataset = await dbPool.executeQuery(querySearch)
			existingDataset = (existingDataset.length>0)? existingDataset[0] : null;
		}

		// INSERT ou UPDATE en fonction ou non d'un enregistrement similaire
		if(!existingDataset) {
			await dbPool.insertData(dataTable, dataSet);
		}
		else {
			// Isoler les données qui ne sont pas déjà dans l'enregistrement de la base
			for(const key of Object.keys(dataSet)) {if(dataSet[key]==existingDataset[key]) delete dataSet[key]};

			if(dataSet.length!=0) {
				// En mode UPDATE, exclure les données vide
				if(mode=="UPDATE") for(const key of Object.keys(dataSet)) {if(dataSet[key]=="") delete dataSet[key]};
				if(mode!="IGNORE") for(const key of Object.keys(dataSet)) {if(existingDataset[key]!="") delete dataSet[key]};
				//console.log("Update dataSet :", dataSet);
				
				if(dataSet.length) {
					//console.log("UPDATE "+dataTable, dataSet);
					await dbPool.updateData(dataTable, dataSet)
				}
			}
		}
		resolve(true);
	})
}


async function getSchema(header, model) {
	//console.log("header:", header);
	// Vérifie si un fichier répond à un schéma reconnu selon le modèle attendu
	return new Promise((resolve, reject)=> {
		// Vérification de la disponibilité du fichier
		let nSchema = 0
		console.log(`> Parcours des schémas du modèle "${model.model}"`);
		while(nSchema<model.schema.length) {
			const allLabelsValid = model.schema[nSchema].structure
				.filter(obj => 'label' in obj) // On de regarde que les colonnes ayant un label
				.every(obj => header.includes(obj.label)); // Le label attendu est-il bien présent dans le header
			if(allLabelsValid) return resolve(model.schema[nSchema]);
			nSchema++;
		}
		reject("!!! Schéma fichier non identifié")
	});
}