/** @preserve Copyright (c) KNOWLEDGECODE - MIT License */
type CompiledObject = string[];
/**
 * Compiles a format string into a tokenized array for efficient parsing and formatting.
 * @param formatString - The format string to compile
 * @returns A compiled array where the first element is the original format string,
 *          followed by tokenized format components
 */
declare const compile: (formatString: string) => CompiledObject;

interface DateLike {
    /**
     * Returns the year of the date.
     */
    getFullYear(): number;
    /**
     * Returns the month of the date (0-11).
     */
    getMonth(): number;
    /**
     * Returns the day of the month (1-31).
     */
    getDate(): number;
    /**
     * Returns the hours of the date (0-23).
     */
    getHours(): number;
    /**
     * Returns the minutes of the date (0-59).
     */
    getMinutes(): number;
    /**
     * Returns the seconds of the date (0-59).
     */
    getSeconds(): number;
    /**
     * Returns the milliseconds of the date (0-999).
     */
    getMilliseconds(): number;
    /**
     * Returns the day of the week (0-6, where 0 is Sunday).
     */
    getDay(): number;
    /**
     * Returns the time value in milliseconds since the Unix epoch (January 1, 1970).
     */
    getTime(): number;
    /**
     * Returns the timezone offset in minutes from UTC.
     */
    getTimezoneOffset(): number;
}

interface LocaleOptions {
    compiledObj: CompiledObject;
    style: 'long' | 'short' | 'narrow';
    case?: 'uppercase' | 'lowercase';
}
interface Locale {
    getLocale: () => string;
    getMonthList: (options: LocaleOptions) => string[];
    getDayOfWeekList: (options: LocaleOptions) => string[];
    getMeridiemList: (options: LocaleOptions) => string[];
}

interface Numeral {
    encode: (str: string) => string;
    decode: (str: string) => string;
}

interface TimeZone {
    zone_name: string;
    gmt_offset: number[];
}

interface FormatterPluginOptions {
    /**
     * The hour format to use for formatting.
     * This is used when the hour is in 12-hour format.
     * It can be 'h11' for 11-hour format or 'h12' for 12-hour format.
     */
    hour12: 'h11' | 'h12';
    /**
     * The hour format to use for formatting.
     * This is used when the hour is in 24-hour format.
     * It can be 'h23' for 23-hour format or 'h24' for 24-hour format.
     */
    hour24: 'h23' | 'h24';
    /**
     * The numeral system to use for formatting numbers.
     * This is an object that provides methods to encode and decode numbers in the specified numeral system.
     */
    numeral: Numeral;
    /**
     * The calendar system to use for formatting dates.
     * This can be 'buddhist' for Buddhist calendar or 'gregory' for Gregorian calendar.
     */
    calendar: 'buddhist' | 'gregory';
    /**
     * The time zone to use for formatting dates and times.
     * This can be a specific time zone object or 'UTC' to use Coordinated Universal Time.
     * If not specified, it defaults to undefined, which means the local time zone will be used.
     */
    timeZone: TimeZone | 'UTC' | undefined;
    /**
     * The locale to use for formatting dates and times.
     * This is an object that provides methods to get localized month names, day names, and meridiems.
     */
    locale: Locale;
}
declare abstract class FormatterPlugin {
    [key: string]: (d: DateLike, options: FormatterPluginOptions, compiledObj: CompiledObject) => string;
}
interface FormatterOptions extends Partial<FormatterPluginOptions> {
    plugins?: FormatterPlugin[];
}

/**
 * Formats a Date object according to the specified format string.
 * @param dateObj - The Date object to format
 * @param arg - The format string or compiled object to match against the Date object
 * @param [options] - Optional formatter options for customization
 * @returns The formatted date string representation
 */
declare function format(dateObj: Date, arg: string | CompiledObject, options?: FormatterOptions): string;

type ParserToken = 'Y' | 'M' | 'D' | 'H' | 'A' | 'h' | 'm' | 's' | 'S' | 'Z';
interface ParserPluginOptions {
    /**
     * The hour format to use for parsing.
     * This is used when the hour is in 12-hour format.
     * It can be 'h11' for 11-hour format or 'h12' for 12-hour format.
     */
    hour12: 'h11' | 'h12';
    /**
     * The hour format to use for parsing.
     * This is used when the hour is in 24-hour format.
     * It can be 'h23' for 23-hour format or 'h24' for 24-hour format.
     */
    hour24: 'h23' | 'h24';
    /**
     * The numeral system to use for parsing numbers.
     * This is an object that provides methods to encode and decode numbers in the specified numeral system.
     */
    numeral: Numeral;
    /**
     * The calendar system to use for parsing dates.
     * This can be 'buddhist' for Buddhist calendar or 'gregory' for Gregorian calendar.
     */
    calendar: 'buddhist' | 'gregory';
    /**
     * Whether to ignore case when matching strings.
     * This is useful for matching month names, day names, and meridiems in a case-insensitive manner.
     * If true, the parser will convert both the input string and the strings in the locale to lowercase before matching.
     */
    ignoreCase: boolean;
    /**
     * The time zone to use for parsing dates and times.
     * This can be a specific time zone object or 'UTC' to use Coordinated Universal Time.
     * If not specified, it defaults to undefined, which means the local time zone will be used.
     */
    timeZone: TimeZone | 'UTC' | undefined;
    /**
     * The locale to use for parsing dates and times.
     * This is an object that provides methods to get localized month names, day names, and meridiems.
     */
    locale: Locale;
}
interface ParseResult {
    value: number;
    length: number;
    token?: ParserToken;
}
declare abstract class ParserPlugin {
    [key: string]: (str: string, options: ParserPluginOptions, compiledObj: CompiledObject) => ParseResult;
}
interface ParserOptions extends Partial<ParserPluginOptions> {
    plugins?: ParserPlugin[];
}

interface PreparseResult {
    /**
     * Year component
     */
    Y?: number;
    /**
     * Month component (1-12)
     */
    M?: number;
    /**
     * Day component
     */
    D?: number;
    /**
     * Hour in 24-hour format
     */
    H?: number;
    /**
     * Meridiem indicator (0:AM/1:PM)
     */
    A?: number;
    /**
     * Hour in 12-hour format
     */
    h?: number;
    /**
     * Minute component
     */
    m?: number;
    /**
     * Second component
     */
    s?: number;
    /**
     * Millisecond component
     */
    S?: number;
    /**
     * Timezone offset in minutes
     */
    Z?: number;
    /**
     * Current parsing position
     */
    _index: number;
    /**
     * Total length of date string
     */
    _length: number;
    /**
     * Number of matched tokens
     */
    _match: number;
}
/**
 * Preparses a date string according to the specified pattern.
 * @param dateString - The date string to preparse
 * @param arg - The pattern string or compiled object to match against the date string
 * @param [options] - Optional parser options
 * @returns PreparseResult containing parsed date components and metadata
 */
declare function preparse(dateString: string, arg: string | CompiledObject, options?: ParserOptions): PreparseResult;

/**
 * Validates whether a date string is valid according to the specified format.
 * @param dateString - The date string to validate
 * @param arg - The format string or compiled object
 * @param [options] - Optional parser options
 * @returns True if the date string is valid, false otherwise
 */
declare function isValid(dateString: string, arg: string | CompiledObject, options?: ParserOptions): boolean;

/**
 * Parses a date string according to the specified format.
 * @param dateString - The date string to parse
 * @param arg - The format string or compiled object to match against the date string
 * @param [options] - Optional parser options for customization
 * @returns The parsed Date object, or an invalid date if parsing fails
 */
declare function parse(dateString: string, arg: string | CompiledObject, options?: ParserOptions): Date;

/**
 * Transforms a date string from one format to another.
 * @param dateString - The date string to transform
 * @param arg1 - The format string or compiled object for parsing
 * @param arg2 - The format string or compiled object for formatting
 * @param [options1] - Optional parser options
 * @param [options2] - Optional formatter options
 * @returns The transformed date string
 */
declare function transform(dateString: string, arg1: string | CompiledObject, arg2: string | CompiledObject, options1?: ParserOptions, options2?: FormatterOptions): string;

/**
 * Adds the specified number of years to a Date object.
 * @param dateObj - The Date object to modify
 * @param years - The number of years to add
 * @param [timeZone] - Optional time zone object or 'UTC'
 * @returns A new Date object with the specified number of years added
 */
declare function addYears(dateObj: Date, years: number, timeZone?: TimeZone | 'UTC'): Date;

/**
 * Adds the specified number of months to a Date object.
 * @param dateObj - The Date object to modify
 * @param months - The number of months to add
 * @param [timeZone] - Optional time zone object or 'UTC'
 * @returns A new Date object with the specified number of months added
 */
declare function addMonths(dateObj: Date, months: number, timeZone?: TimeZone | 'UTC'): Date;

/**
 * Adds the specified number of days to a Date object.
 * @param dateObj - The Date object to modify
 * @param days - The number of days to add
 * @param [timeZone] - Optional time zone object or 'UTC'
 * @returns A new Date object with the specified number of days added
 */
declare function addDays(dateObj: Date, days: number, timeZone?: TimeZone | 'UTC'): Date;

/**
 * Adds the specified number of hours to a Date object.
 * @param dateObj - The Date object to modify
 * @param hours - The number of hours to add
 * @returns A new Date object with the specified number of hours added
 */
declare function addHours(dateObj: Date, hours: number): Date;

/**
 * Adds the specified number of minutes to a Date object.
 * @param dateObj - The Date object to modify
 * @param minutes - The number of minutes to add
 * @returns A new Date object with the specified number of minutes added
 */
declare function addMinutes(dateObj: Date, minutes: number): Date;

/**
 * Adds the specified number of seconds to a Date object.
 * @param dateObj - The Date object to modify
 * @param seconds - The number of seconds to add
 * @returns A new Date object with the specified number of seconds added
 */
declare function addSeconds(dateObj: Date, seconds: number): Date;

/**
 * Adds the specified number of milliseconds to a Date object.
 * @param dateObj - The Date object to modify
 * @param milliseconds - The number of milliseconds to add
 * @returns A new Date object with the specified number of milliseconds added
 */
declare function addMilliseconds(dateObj: Date, milliseconds: number): Date;

interface NanosecondsParts {
    nanoseconds: number;
}
interface MicrosecondsParts extends NanosecondsParts {
    microseconds: number;
}
interface MillisecondsParts extends MicrosecondsParts {
    milliseconds: number;
}
interface SecondsParts extends MillisecondsParts {
    seconds: number;
}
interface MinutesParts extends SecondsParts {
    minutes: number;
}
interface HoursParts extends MinutesParts {
    hours: number;
}
interface DaysParts extends HoursParts {
    days: number;
}
interface DurationDescriptor<T> {
    /**
     * The value of the duration in the respective unit.
     */
    value: number;
    /**
     * Formats the duration according to the provided format string.
     * @param formatString - The format string to use for formatting
     * @param numeral - Optional numeral object for number formatting
     * @returns Formatted string representation of the duration
     */
    format: (formatString: string, numeral?: Numeral) => string;
    /**
     * Converts the duration to an object containing the parts of the duration in the respective unit.
     * @returns An object containing the parts of the duration in the respective unit.
     */
    toParts: () => T;
}
declare class Duration {
    private readonly time;
    constructor(time: number | DOMHighResTimeStamp);
    /**
     * Converts the duration to nanoseconds.
     * @returns DurationDescriptor with the value in nanoseconds and methods to format and get
     * the parts of the duration in nanoseconds.
     */
    toNanoseconds(): DurationDescriptor<NanosecondsParts>;
    /**
     * Converts the duration to microseconds.
     * @returns DurationDescriptor with the value in microseconds and methods to format and get
     * the parts of the duration in microseconds and nanoseconds.
     */
    toMicroseconds(): DurationDescriptor<MicrosecondsParts>;
    /**
     * Converts the duration to milliseconds.
     * @returns DurationDescriptor with the value in milliseconds and methods to format and get
     * the parts of the duration in milliseconds, microseconds, and nanoseconds.
     */
    toMilliseconds(): DurationDescriptor<MillisecondsParts>;
    /**
     * Converts the duration to seconds.
     * @returns DurationDescriptor with the value in seconds and methods to format and get
     * the parts of the duration in seconds, milliseconds, microseconds, and nanoseconds.
     */
    toSeconds(): DurationDescriptor<SecondsParts>;
    /**
     * Converts the duration to minutes.
     * @returns DurationDescriptor with the value in minutes and methods to format and get
     * the parts of the duration in minutes, seconds, milliseconds, microseconds, and nanoseconds.
     */
    toMinutes(): DurationDescriptor<MinutesParts>;
    /**
     * Converts the duration to hours.
     * @returns DurationDescriptor with the value in hours and methods to format and get
     * the parts of the duration in hours, minutes, seconds, milliseconds, microseconds, and nanoseconds.
     */
    toHours(): DurationDescriptor<HoursParts>;
    /**
     * Converts the duration to days.
     * @returns DurationDescriptor with the value in days and methods to format and get
     * the parts of the duration in days, hours, minutes, seconds, milliseconds, microseconds, and nanoseconds.
     */
    toDays(): DurationDescriptor<DaysParts>;
}

/**
 * Calculates the difference between two dates.
 * @param from - The first Date object
 * @param to - The second Date object
 * @returns A Duration instance with methods to get the difference in various units
 */
declare const subtract: (from: Date, to: Date) => Duration;

/**
 * Determines if the specified year is a leap year.
 * @param year - The year to check
 * @returns True if the year is a leap year, false otherwise
 */
declare const isLeapYear: (year: number) => boolean;
/**
 * Determines if two dates represent the same calendar day.
 * @param date1 - The first date to compare
 * @param date2 - The second date to compare
 * @returns True if both dates are on the same day, false otherwise
 */
declare const isSameDay: (date1: Date, date2: Date) => boolean;

export { Duration, addDays, addHours, addMilliseconds, addMinutes, addMonths, addSeconds, addYears, compile, format, isLeapYear, isSameDay, isValid, parse, preparse, subtract, transform };
