import fs from 'node:fs';
import { fileURLToPath } from 'url';
import { dirname, join } from 'path';
import { WSServer } from './modules/zWebSocketServer.mjs';
import { PoolMariaDB } from './modules/zPoolMariaDB.mjs';
import { fileImportuseDbPool, importFile } from './modules/zFileImport.mjs';
import { TaskManager } from './modules/zTaskManager.mjs';

// Lecture des paramètres de connexion à la DB
import dbConfig from './data/db.json' with { type: "json" };

// Lecture des paramètres contenu dans le fichier webAPI.json
import config from './data/webAPI.json' with { type: "json" };

// Récupération du chemin du script (pour localiser le répertoire d'upload)
const __filename = fileURLToPath(import.meta.url);
const __dirname = dirname(__filename);

// Création du WebServer 
const ws = new WSServer({
	port: config.webService.port,
	onMessage: ({ clientId, message }) => {
		//console.log(`📨 Message de ${clientId}:`, message);
		console.log(`📨 Message de ${clientId}`);
		messageProcessing(clientId, message);
	}
});

// Création du Pool de connexion à la DB
const infoTech = new PoolMariaDB(dbConfig);
// Passage du pool aux fonction d'import
fileImportuseDbPool(infoTech);

// Création du gestionnaire de tâche pour le séquencage des appels des fonctions sollicitant la DB
const taskManager = new TaskManager();

/*=============================================
		TRAITEMENT DES MESSAGES ENTRANTS
	=============================================*/
async function messageProcessing(clientId, message) {
	if(message.type) {
		switch(message.type.toUpperCase()) {
			case "IMPORT FILES":
				console.log(" - Demande d'import fichier(s): ");
				filesImport(message.payload, clientId);
				break;
			case "PING":
				setTimeout(() => {
					ws.sendToClient(clientId,{type:"PING", payload:"PONG"});
				}, 2000);
				break;
			case "TEST":
				setTimeout(() => {
					ws.sendToClient(clientId,{type:"any", payload:message.payload});
				}, 2000);
				break;
			default:
				console.error(`‼️Type message inconnu [${message.type}]`);
		}
	} else console.error("‼️Absence de 'type' dans le message\n", message);

}

/*============================================*/
async function filesImport(data, clientId) {
	// Contrôle du contenu
	if(!data.files) throw new Error("La demande d'import ne contient pas de fichiers");
	const files = JSON.parse(data.files);
	files.map(file => {
		const param = {file:file, model:data.model, dbTable:data.dbTable, mode:data.mode, for:clientId};
		taskManager.pushTask(() => trtImport(param));
	});
}

async function trtImport(obj) {
	return new Promise((resolve, reject) => {
		// Appel traitement d'import et gestion des message renvoyé au client demandeur
		const relativePath = "../uploads/"+obj.file.fileName;
		const filePath = join(__dirname, relativePath);
		importFile({filePath:filePath, model:obj.model, dbTable:obj.dbTable, mode:obj.mode})
		.then(CR => { 
			ws.sendToClient(obj.for,{type:"any", payload:CR });
			resolve(CR);
		})
		.catch(err => {
			ws.sendToClient(obj.for,{type:"any", payload:err});
			reject(err);
		})
		.finally(() => {
			// Supression du fichier uploadé
			fs.unlink(filePath, (err) => {
				if (err) console.error("Error:",err);
			});
		})
	})
}
