/*=========================================================
	MODULE zJsLibrary
	----------------------------------------------------
	Bibliothèque de gestion des boites de dialogue
=========================================================*/
import { FormManager } from "./zJsForms.js";

export class Dialog {
	static #instances = new Set();

	constructor(objData) {
		if (objData.persistant) this.persistant=true;

		// Construction Dialog
		const dialog = document.createElement("dialog");							// Création de la boite de dialogue
		if(objData.id) dialog.setAttribute('id', objData.id);
		if(objData.type) dialog.classList.add(objData.type);
		dialog.style.zIndex = 999;

		const titleBar = document.createElement('div');								// Création de la la barre de Titre
		titleBar.classList = "zDialTitleBar";
		if(objData.icon) {
			const icon = document.createElement('img');
			icon.src = "../media/"+objData.icon;
			icon.classList = "zDialIcon";
			titleBar.appendChild(icon);
		}
		const title = document.createElement('span');								// Création du texte du Titre
		title.classList = "zDialTitle unselectable";
		title.innerHTML = objData.title;
		this.titleDiv = title;
		titleBar.appendChild(title);														// Ajout du texte dans la barre de Titre

		if (!objData.noSysClose) {                                               
			const sysClose = document.createElement('img');							// Création de l'image Bouton de Fermeture
			sysClose.classList.add('zDialSysClose');
			sysClose.setAttribute('src', '../media/checkbox_cross.png');
			sysClose.addEventListener("click", () => this.close());
			titleBar.appendChild(sysClose);												// Ajout de l'image Bouton de Fermeture à la barre de Titre
		}
		dialog.appendChild(titleBar);														// Ajout de la barre de Titre à la boite de dialogue

		const primeContainer = document.createElement('div');						// Création du Conteneur principal
		primeContainer.classList = "zDialPrimeContainer";
		
		if(objData.type) {
			const signContainer = document.createElement('div');
			signContainer.classList = "zObjectContainer zHCenter zVCenter";
			const signImage = document.createElement('img');
			signImage.classList = "zDialSign";
			signImage.src = `../media/dial_${objData.type}.png`

			signContainer.appendChild(signImage);
			primeContainer.appendChild(signContainer);
		} 

		const mainContainer = document.createElement('div');							// Création du Conteneur Générique
		mainContainer.classList = "zFlexCol";

		if(objData.content) mainContainer.appendChild(objData.content)				// Ajout de l'élément 'content'
		if(objData.form) {
			const dialForm = new FormManager(objData.form);
			mainContainer.appendChild(dialForm.html)										// Ajout de l'élément 'content'
		}

		if(objData.message) {
			const messageContainer = document.createElement('div');					// Création de la div Message
			messageContainer.classList = "zObjectContainer";
			messageContainer.innerHTML = objData.message;
			mainContainer.appendChild(messageContainer);									// Ajout de la div Message au Conteneur Générique
		}
		if (objData.uploadContainer) {
			// Structure du dialogue d'import Autodesk
			const uploadContainer		= document.createElement('div');
			uploadContainer.classList	= "zObjectContainer zLabelUp";
			const span						= document.createElement('span');
			span.innerHTML					= objData.uploadContainer.label;
			const inputFile				= document.createElement('input');
			inputFile.id					= "uploadFiles"
			inputFile.type					= "file"
			inputFile.setAttribute('name', "uploadFiles");
			inputFile.setAttribute('accept', objData.uploadContainer.formats);
			if (objData.uploadContainer.multiple) inputFile.setAttribute('multiple', true);
			uploadContainer.appendChild(span);
			uploadContainer.appendChild(inputFile);
			mainContainer.appendChild(uploadContainer);
		}

		if (objData.buttons) {
			const buttonContainer = document.createElement('div');					// Création de la div des boutons options
			buttonContainer.classList = "zFlexRow";
			const spacer = document.createElement('div');
			spacer.classList = "zObjectContainer zFGrow";
			buttonContainer.appendChild(spacer);

			objData.buttons.forEach(button => {
				const objContainer = document.createElement('div');
				objContainer.classList = "zObjectContainer";
				const btn = document.createElement('button');							// Création du bouton
				btn.classList='zButton';
				if (button.id) btn.id = button.id;
				if (button['icon']) {
					btn.classList.add("zButtonImageStd");
					const img = document.createElement('img');							// Création de l'icone du bouton
					img.classList.add('zIconzButtonImageLeft');
					img.setAttribute('src', `../media/${button['icon']}`)
					img.setAttribute('alt', ``)
					btn.appendChild(img);
				} else btn.classList.add("zButtonMenuStd");
				btn.classList.add("zMgL");

				if (button.onClick) btn.setAttribute('onclick', button.function);
				if (button.cancel) btn.addEventListener("click", () => this.close());
				btn.append(button['lib']);
				objContainer.appendChild(btn);
				buttonContainer.appendChild(objContainer);								// Ajout du bouton à la div des boutons options
			})

			mainContainer.appendChild(buttonContainer);									// Ajout de la div Boutons options au Conteneur Générique

		}

		primeContainer.appendChild(mainContainer);
		dialog.appendChild(primeContainer);													// Ajout du Conteneur Générique à la boite de dialogue
		this.html = dialog;

		document.querySelector('body').appendChild(dialog);

		Dialog.#instances.add(this);
		
		switch (objData.onCreation) {
			case "open":
				this.open(false);
				break;
			case "openModal":
				this.open(true);
				break
		}
	
	}

	onButton(buttonId, handler) {
		let btnElm = this.html.querySelector(`#${buttonId}`);
		if(btnElm) {
			console.log(`Set btn id:${buttonId} 'click' handler`);
			btnElm.addEventListener('click', () => {handler()});
		}
	}

	set title(title) {
		this.titleDiv.innerHTML = title;
	}

	static getInstances() {
		return Array.from(Dialog.#instances);
	}

	open(modal=false) {
		if(modal) {
			this.html.showModal();
		} else {
			this.html.show();
			this.#draggable();
		}
		// Centrage de la boite de dialogue dans la fenêtre navigateur
		this.html.style.left = (window.innerWidth-this.html.offsetWidth)/2+'px';
		this.html.style.top  = (window.innerHeight-this.html.offsetHeight)/2+'px';

		// La boite de dialogue est visible
		this.viewable = true;
	}

	onClose(handler) {
		this.html.addEventListener('close', () => {
			//console.log("Dial close listener");
			handler(true);
		});
	}

	close() {
		this.html.close();
		this.viewable = false;
		if (!this.persistant) this.remove();
	}

	remove() {
		//this.html.parentNode.removeChild(this.html);
		/* 1. Retire le nœud du DOM */
		if (this.html?.isConnected) {
			this.html.remove();
		}

		/* 2. Détache les écouteurs éventuels (bon pour éviter les fuites) */
		//  … si vous avez enregistré d’autres listeners, retirez‑les ici …

		/* 3. Efface l’instance de la collection */
		Dialog.#instances.delete(this);

		/* 4. Aide facultative au GC : on « vide » les refs */
		for (const k of Object.keys(this)) {
			this[k] = null;
		}
	}

	#draggable() {
		const element = this.html;

		let deltaX = 0, deltaY = 0, prevX = 0, prevY = 0;

		const header = this.html.querySelector("div.zDialTitleBar");
		header.classList.add("draggable");
		const btnClose = this.html.querySelector("img.zDialSysClose");

		header.addEventListener("mousedown", dragMouseDown);
		btnClose.addEventListener("click", closeDialog);

		function dragMouseDown(e) {
			e.preventDefault();
			prevX = e.clientX;
			prevY = e.clientY;
			document.addEventListener("mouseup", closeDragElement);
			document.addEventListener("mousemove", elementDrag);
		}

		function closeDialog() {
			//console.log(">>> Fermeture de la fenêtre de dialogue");
			document.removeEventListener("mouseup", closeDragElement);
			document.removeEventListener("mousemove", elementDrag);
			header.removeEventListener("mousedown",dragMouseDown);
			btnClose.removeEventListener("click", closeDialog);
		}

		function elementDrag(e) {
			e.preventDefault();
			deltaX = prevX - e.clientX;
			deltaY = prevY - e.clientY;
			prevX = e.clientX;
			prevY = e.clientY;
			element.style.top = element.offsetTop - deltaY + "px";
			element.style.left = element.offsetLeft - deltaX + "px";
		}

		function closeDragElement() {
			document.removeEventListener("mouseup", closeDragElement);
			document.removeEventListener("mousemove", elementDrag);
		}

		function resetPosition() {
			element.style.left = '0px';
			element.style.top = '0px';
		}
	}

}
